#include "stdafx.h"
#include "ModelManager.h"

CModel::CModel()
	: m_model(nullptr), m_modelInstance(nullptr), m_file(nullptr), m_fileInfo(nullptr)
{
}

CModel::~CModel()
{
	Destroy();
}

bool CModel::Load(const fs::path& path)
{
	if (!fs::is_regular_file(path))
		return false;

	granny_file* m_file = GrannyReadEntireFile(path.string().c_str());
	granny_file_info* m_fileInfo = GrannyGetFileInfo(m_file);
	granny_file_info* info = m_fileInfo;

	assert(1 == info->ModelCount);

	m_model = info->Models[0];

	GrannyFreeFileSection(m_file, GrannyStandardRigidVertexSection);
	GrannyFreeFileSection(m_file, GrannyStandardRigidIndexSection);
	GrannyFreeFileSection(m_file, GrannyStandardDeformableIndexSection);
	GrannyFreeFileSection(m_file, GrannyStandardTextureSection);

	return true;
}

void CModel::Destroy()
{
	if (m_file)
	{
		GrannyFreeFile(m_file);
		m_file = 0;
	}
}

bool CModel::IsGrannyFile(const fs::path& path)
{
	if (false == fs::is_regular_file(path))
		return false;

	std::string ext = path.extension().string();
	std::for_each(ext.begin(), ext.end(), [](char& c) { c = ::tolower(c); });
	return ext == ".gr2";
}

bool CModel::IsGrannyModelFile(const fs::path& path)
{
	if (!fs::is_regular_file(path))
		return false;

	granny_file* file = GrannyReadEntireFile(path.string().c_str());
	assert(file && "Failed to open granny file");

	granny_file_info* info = GrannyGetFileInfo(file);
	assert(file && "Failed to read granny file info");

	const bool bResult = 0 < info->ModelCount;

	GrannyFreeFile(file);

	return bResult;
}

CModelManager::CModelManager()
{
}

CModelManager::~CModelManager()
{
	Destroy();
}

CModel* CModelManager::AutoRegisterAndGetModel(const fs::path& initPath, int32_t depth)
{
	fs::path curPath = initPath;

	while (depth--)
	{
		curPath = curPath.parent_path();

		for (fs::directory_iterator endIter, iter(curPath); iter != endIter; ++iter)
		{
			const fs::path& path = iter->path();

			if (CModel::IsGrannyFile(path) && CModel::IsGrannyModelFile(path))
			{
				return this->RegisterModel(path);
			}
		}
	}

	return nullptr;
}

CModel* CModelManager::RegisterModel(const fs::path& path)
{
	if (false == fs::is_regular_file(path))
		return nullptr;

	const std::string key = path.parent_path().string();

	CModel* model = GetModel(key);

	if (0 != model)
		return model;

	model = new CModel();

	if (model->Load(path))
	{
		m_modelMap.insert(std::make_pair(key, model));
		return model;
	}

	delete model;

	return nullptr;
}

void CModelManager::Destroy()
{
	for (auto & iter : m_modelMap)
	{
		CModel* model = iter.second;

		model->Destroy();
		delete model;
	}

	m_modelMap.clear();

}

CModel* CModelManager::GetModel(const fs::path& path)
{
	const std::string key = path.parent_path().string();

	TModelCache::iterator iter = m_modelMap.find(key);

	if (m_modelMap.end() == iter)
		return nullptr;

	return iter->second;
}
